/*
 * Copyright (c) Haulmont 2023. All Rights Reserved.
 * Use is subject to license terms.
 */

function FormConnector(formEditor) {
    const connector = this;

    const selection = formEditor.get('selection');
    const formFieldRegistry = formEditor.get('formFieldRegistry');
    const eventBus = formEditor.get('eventBus');
    const modeling = formEditor.get('modeling');

    const VALUES_SOURCES = {
        EXPRESSION: "expression",
        INPUT: "input",
        STATIC: "static"
    }

    const VALUES_SOURCES_PATHS = {
        expression: ['valuesExpression'], input: ['valuesKey'], static: ['values']
    }

    const VALUES_SOURCES_DEFAULTS = {
        expression: "=",
        input: "",
        static: [{label: 'Value', value: 'value'}]
    }

    let silentMode = false;

    function doSilently(action) {
        silentMode = true;
        try {
            action();
        } finally {
            silentMode = false;
        }
    }

    function on(eventType, handler) {
        eventBus.on(eventType, (event) => {
            if (!silentMode)
                handler(event);
        });
    }

    connector.setFromJson = function (formJson) {
        doSilently(() => {
            let selectedElement = selection.get();
            if (selectedElement) {
                let selectedId = selectedElement["id"];
                formEditor.importSchema(formJson)
                    .then(function () {
                        selection.set(formFieldRegistry.get(selectedId));
                    })
                    .catch(function (err) {
                        console.error('could not import from diagram', err);
                    })
            } else {
                formEditor.importSchema(formJson)
            }
        })
    };

    connector.reselectElement = function () {
        sendCefQuery("ELEMENT_SELECTED", {'element': selection.get(), 'requestor': 'formJs'});
    }

    connector.selectElement = function (elementId) {
        doSilently(() => {
            selection.clear();
            selection.set(formFieldRegistry.get(elementId));
        });
        sendCefQuery("ELEMENT_SELECTED", {'element': selection.get(), 'requestor': 'connector'});

    }

    connector.updateFormElementProperty = function (cmdJson) {
        const {
            elementId,
            keepBlank,
            ...restProps
        } = JSON.parse(cmdJson);

        let element = formFieldRegistry.get(elementId);

        if (!keepBlank) {
            for (let p in restProps) {
                if (restProps[p] === "") {
                    restProps[p] = undefined
                }
            }
        }

        modeling.editFormField(element, restProps)
    }

    connector.updateFormElementPropertyByPath = function (cmdJson) {
        const {
            elementId,
            path,
            ...restProps
        } = JSON.parse(cmdJson);

        let element = formFieldRegistry.get(elementId);

        let prop = element[path[0]]

        if (!prop) {
            prop = {}
        }

        for (let p in restProps) {
            prop[p] = restProps[p]
        }

        modeling.editFormField(element, path, prop)
    }

    connector.updateOptionSourceProperty = function (cmdJson) {
        const {
            elementId,
            optionSourceType
        } = JSON.parse(cmdJson);

        let element = formFieldRegistry.get(elementId);

        const newProperties = {};

        Object.values(VALUES_SOURCES).forEach(source => {

            // Clear all values source definitions and default the newly selected one
            const newValue = optionSourceType === source ? VALUES_SOURCES_DEFAULTS[source] : undefined;
            newProperties[VALUES_SOURCES_PATHS[source]] = newValue;
        });

        modeling.editFormField(element, newProperties)
    }

    connector.updateValidationPatternProperty = function (cmdJson) {
        const {
            elementId,
            patternType
        } = JSON.parse(cmdJson);

        let element = formFieldRegistry.get(elementId);

        const validate = element['validate'] || {};

        let property;
        if (patternType === 'custom') {
            property = undefined;
        } else {
            property = patternType;
        }

        validate['validationType'] = property;

        modeling.editFormField(element, ['validate'], validate)
    }

    on("selection.changed", function (e) {
        sendCefQuery("ELEMENT_SELECTED", {'element': e.selection, 'requestor': 'formJs'});
    });

    on("changed", function (e) {
        sendCefQuery("SCHEMA_UPDATE", formEditor.saveSchema());
    });

    on("formEditor.rendered", function (e) {
        let payload = '';
        if (formFieldRegistry) {
            let elements = formFieldRegistry.getAll();
            if (elements && elements.length > 0) {
                payload = elements[0]['id'];
            }
        }
        sendCefQuery("FORM_INIT", payload);
    });

    (async function () {
        sendCefQuery("INITIALIZATION");
    })();

    formEditor.get('propertiesPanel').detach();
}

function sendCefQuery(id, payload) {
    if (payload == null)
        payload = "";

    let query = {
        request: JSON.stringify(
            {
                "id": id,
                "payload": (typeof payload === "string" || payload instanceof String) ? payload : JSON.stringify(payload)
            }
        ),
        onSuccess: function (response) {
            console.log(response);
        },
        onFailure: function (error_code, error_message) {
            console.log("Error: (" + error_code + ") " + error_message);
        }
    };

    window.cefQueryBpm(query);
}